const { app, BrowserWindow, ipcMain, dialog } = require('electron');
const path = require('path');
const drivelist = require('drivelist');
const { exec } = require('child_process');
const util = require('util');
const execPromise = util.promisify(exec);

let mainWindow;

function createWindow() {
  mainWindow = new BrowserWindow({
    width: 1400,
    height: 900,
    webPreferences: {
      nodeIntegration: true,
      contextIsolation: false,
      enableRemoteModule: true
    },
    icon: path.join(__dirname, '../assets/icon.png'),
    title: 'PPSP Data Destruction',
    darkTheme: true
  });

  mainWindow.loadFile('src/index.html');

  // Uncomment to open DevTools
  // mainWindow.webContents.openDevTools();
}

app.whenReady().then(createWindow);

app.on('window-all-closed', () => {
  if (process.platform !== 'darwin') app.quit();
});

app.on('activate', () => {
  if (BrowserWindow.getAllWindows().length === 0) createWindow();
});

// IPC Handlers

// Get all connected drives
ipcMain.handle('get-drives', async () => {
  try {
    const drives = await drivelist.list();
    return { success: true, drives };
  } catch (error) {
    return { success: false, error: error.message };
  }
});

// Wipe drive with selected method
ipcMain.handle('wipe-drive', async (event, { device, method, passes }) => {
  try {
    // Safety check - prevent wiping system drive
    const drives = await drivelist.list();
    const targetDrive = drives.find(d => d.device === device);

    if (!targetDrive) {
      return { success: false, error: 'Drive not found' };
    }

    // Check if it's a system drive (macOS)
    if (process.platform === 'darwin') {
      const { stdout } = await execPromise('mount | grep "on / "');
      if (stdout.includes(device)) {
        return { success: false, error: 'Cannot wipe system drive!' };
      }
    }

    // Simulate wipe operation (replace with actual implementation)
    const result = await simulateWipe(device, method, passes);

    return { 
      success: true, 
      message: `Drive ${device} wiped successfully using ${method}`,
      details: result
    };

  } catch (error) {
    return { success: false, error: error.message };
  }
});

// Simulate wipe operation (replace with actual disk utility calls)
async function simulateWipe(device, method, passes) {
  const startTime = Date.now();

  // In production, replace with actual disk wiping commands:
  // macOS: diskutil secureErase
  // Linux: shred or dd
  // Windows: format or sdelete

  const methods = {
    'quick': { name: 'Quick Wipe', passes: 1, description: 'Single pass of zeros' },
    'dod-5220': { name: 'DoD 5220.22-M', passes: 3, description: 'DoD standard 3-pass' },
    'dod-ece': { name: 'DoD 5220.22-M ECE', passes: 7, description: 'DoD Extended 7-pass' },
    'gutmann': { name: 'Gutmann Method', passes: 35, description: '35-pass maximum security' },
    'nist': { name: 'NIST 800-88', passes: 1, description: 'NIST Clear standard' },
    'vsitr': { name: 'VSITR', passes: 7, description: 'German VSITR 7-pass' },
    'random': { name: 'Random Data', passes: 3, description: '3-pass random data' },
    'ata-secure': { name: 'ATA Secure Erase', passes: 1, description: 'Hardware-level SSD erase' }
  };

  const selectedMethod = methods[method] || methods['quick'];

  // Simulate progress
  for (let i = 1; i <= selectedMethod.passes; i++) {
    await new Promise(resolve => setTimeout(resolve, 1000)); // Simulate work
    mainWindow.webContents.send('wipe-progress', {
      pass: i,
      totalPasses: selectedMethod.passes,
      percent: Math.round((i / selectedMethod.passes) * 100)
    });
  }

  const endTime = Date.now();

  return {
    method: selectedMethod.name,
    passes: selectedMethod.passes,
    duration: ((endTime - startTime) / 1000).toFixed(2),
    timestamp: new Date().toISOString(),
    device: device
  };
}

// Generate certificate
ipcMain.handle('generate-certificate', async (event, data) => {
  const certificate = {
    id: 'CERT-' + Date.now(),
    company: 'PPSP Recycling LLC',
    date: new Date().toLocaleDateString(),
    device: data.device,
    method: data.method,
    serialNumber: data.serialNumber || 'N/A',
    technician: data.technician || 'System',
    standard: getStandardName(data.method),
    verified: true
  };

  return { success: true, certificate };
});

function getStandardName(method) {
  const standards = {
    'quick': 'N/A',
    'dod-5220': 'DoD 5220.22-M',
    'dod-ece': 'DoD 5220.22-M ECE',
    'gutmann': 'Gutmann Method',
    'nist': 'NIST 800-88',
    'vsitr': 'VSITR (German)',
    'random': 'Random Data 3-pass',
    'ata-secure': 'ATA Secure Erase'
  };
  return standards[method] || 'Unknown';
}

// Show save dialog
ipcMain.handle('show-save-dialog', async (event, options) => {
  const result = await dialog.showSaveDialog(mainWindow, options);
  return result;
});