const { app, BrowserWindow, ipcMain, dialog } = require('electron');
const path = require('path');
const drivelist = require('drivelist');
const { exec, spawn } = require('child_process');
const fs = require('fs');
const os = require('os');

let mainWindow;

function createWindow() {
  mainWindow = new BrowserWindow({
    width: 1400,
    height: 900,
    webPreferences: {
      nodeIntegration: true,
      contextIsolation: false
    },
    icon: path.join(__dirname, '../assets/icon.ico'),
    title: 'PPSP Secure Wipe',
    backgroundColor: '#0f172a'
  });
  mainWindow.loadFile('src/index.html');
}

app.whenReady().then(createWindow);
app.on('window-all-closed', () => { if (process.platform !== 'darwin') app.quit(); });

// Get Windows drives
ipcMain.handle('get-drives', async () => {
  try {
    const drives = await drivelist.list();
    const processed = drives.map(d => {
      const isSystem = d.mountpoints.some(mp => mp.path === 'C:\\' || mp.path.startsWith('C:'));
      return {
        device: d.device,
        description: d.description,
        size: d.size,
        letter: d.mountpoints[0]?.path || 'Unknown',
        isSystem: isSystem,
        isUSB: d.isUSB,
        displayName: d.mountpoints[0]?.label || d.description || 'Unknown'
      };
    });
    return { success: true, drives: processed };
  } catch (error) {
    return { success: false, error: error.message };
  }
});

// Windows wipe using cipher or format
ipcMain.handle('wipe-drive', async (event, { device, method }) => {
  try {
    const drives = await drivelist.list();
    const target = drives.find(d => d.device === device);
    if (!target) throw new Error('Drive not found');
    if (target.isSystem) throw new Error('Cannot wipe system drive (C:)!');

    const methodConfig = getMethodConfig(method);
    const driveLetter = target.mountpoints[0]?.path || device;

    return new Promise((resolve, reject) => {
      const startTime = Date.now();

      // Use Windows cipher /w for secure wiping
      // Format: cipher /w:C:\ (writes zeros to free space)
      // For full drive wipe, we use format with /P:passes

      let command;
      if (method === 'quick') {
        // Quick format
        command = `format ${driveLetter} /FS:NTFS /Q /Y`;
      } else {
        // Secure format with multiple passes
        const passes = methodConfig.passes;
        command = `format ${driveLetter} /FS:NTFS /P:${passes} /Y`;
      }

      mainWindow.webContents.send('wipe-progress', {
        pass: 1, totalPasses: methodConfig.passes, percent: 10, status: 'Formatting...'
      });

      exec(command, { windowsHide: true }, (error, stdout, stderr) => {
        if (error) {
          // Try alternative: cipher
          const cipherCmd = `cipher /w:${driveLetter}\\`;
          exec(cipherCmd, { windowsHide: true }, (err2, out2, err2str) => {
            if (err2) {
              reject(new Error(`Wipe failed: ${error.message}`));
            } else {
              const duration = ((Date.now() - startTime) / 1000).toFixed(2);
              resolve({
                success: true,
                details: {
                  method: methodConfig.name,
                  passes: methodConfig.passes,
                  duration: duration,
                  timestamp: new Date().toISOString(),
                  device: device,
                  letter: driveLetter
                }
              });
            }
          });
        } else {
          const duration = ((Date.now() - startTime) / 1000).toFixed(2);
          resolve({
            success: true,
            details: {
              method: methodConfig.name,
              passes: methodConfig.passes,
              duration: duration,
              timestamp: new Date().toISOString(),
              device: device,
              letter: driveLetter
            }
          });
        }
      });
    });
  } catch (error) {
    return { success: false, error: error.message };
  }
});

function getMethodConfig(method) {
  const methods = {
    'quick': { name: 'Quick Format', passes: 0 },
    'nist': { name: 'NIST 800-88', passes: 1 },
    'random': { name: 'Random Data', passes: 3 },
    'dod-5220': { name: 'DoD 5220.22-M', passes: 3 },
    'vsitr': { name: 'VSITR', passes: 7 },
    'dod-ece': { name: 'DoD ECE', passes: 7 },
    'gutmann': { name: 'Gutmann', passes: 35 }
  };
  return methods[method] || methods['quick'];
}

// Certificate
ipcMain.handle('generate-certificate', async (event, data) => {
  const cert = {
    id: 'PPSP-' + Date.now(),
    company: 'PPSP Recycling LLC',
    address: '9095 Elk Grove Blvd Suite B, Elk Grove, CA 95624',
    phone: '916-381-8304',
    date: new Date().toLocaleString(),
    device: data.device,
    letter: data.letter,
    method: data.method,
    standard: data.method,
    duration: data.duration + 's',
    technician: os.userInfo().username,
    platform: 'Windows',
    verified: true
  };
  return { success: true, certificate: cert };
});

ipcMain.handle('save-certificate', async (event, cert) => {
  const result = await dialog.showSaveDialog(mainWindow, {
    defaultPath: `PPSP-Cert-${cert.id}.txt`,
    filters: [{ name: 'Text', extensions: ['txt'] }, { name: 'JSON', extensions: ['json'] }]
  });
  if (!result.canceled) {
    const content = `
PPSP DATA DESTRUCTION CERTIFICATE
=================================
ID: ${cert.id}
Date: ${cert.date}
Company: ${cert.company}
Address: ${cert.address}
Phone: ${cert.phone}

Device: ${cert.device} (${cert.letter})
Method: ${cert.standard}
Duration: ${cert.duration}
Platform: ${cert.platform}
Technician: ${cert.technician}

Status: VERIFIED
`;
    fs.writeFileSync(result.filePath, content);
    return { success: true };
  }
  return { success: false };
});
