# SECURITY HARDENING GUIDE
## Securing PPSP Web Portal & Applications

### Current Security Status

Your web portal is currently deployed to a **Kimi temporary URL**:
- URL: `https://[random-string].ok.kimi.link`
- Security: HTTPS + random subdomain (security through obscurity)
- Authentication: Supabase Auth with email/password

### Recommended Security Enhancements

#### 1. Add Basic Authentication (Immediate - 5 minutes)

Add to your Next.js middleware:

```typescript
// middleware.ts
import { NextResponse } from 'next/server';
import type { NextRequest } from 'next/server';

export function middleware(request: NextRequest) {
  // Check for basic auth header
  const basicAuth = request.headers.get('authorization');

  if (basicAuth) {
    const authValue = basicAuth.split(' ')[1];
    const [user, pwd] = atob(authValue).split(':');

    if (user === 'ppsp' && pwd === 'YourSecurePassword123!') {
      return NextResponse.next();
    }
  }

  // Return 401 if no auth
  return new NextResponse('Authentication required', {
    status: 401,
    headers: {
      'WWW-Authenticate': 'Basic realm="PPSP Secure Area"'
    }
  });
}

export const config = {
  matcher: ['/((?!api|_next/static|_next/image|favicon.ico).*)']
};
```

#### 2. IP Whitelisting (Enterprise)

```typescript
// middleware.ts with IP check
const ALLOWED_IPS = [
  '203.0.113.0/24',    // Your office IP range
  '198.51.100.1',      // Specific IP
  '::1'                // localhost for testing
];

function isIpAllowed(ip: string): boolean {
  return ALLOWED_IPS.some(allowed => {
    if (allowed.includes('/')) {
      // CIDR notation check
      return ipInCidr(ip, allowed);
    }
    return ip === allowed;
  });
}
```

#### 3. Deploy to Private Server (Maximum Security)

**Option A: Your Own Server**
```bash
# Build locally
npm run build

# Deploy to your server
rsync -avz --delete dist/ user@your-server:/var/www/ppsp-portal/

# Configure nginx with client certificates
sudo nano /etc/nginx/sites-available/ppsp-portal
```

**Nginx Config with Client Certs:**
```nginx
server {
    listen 443 ssl;
    server_name portal.ppsprecyclingllc.com;

    ssl_certificate /etc/ssl/certs/server.crt;
    ssl_certificate_key /etc/ssl/private/server.key;
    ssl_client_certificate /etc/ssl/certs/ca.crt;
    ssl_verify_client on;

    location / {
        proxy_pass http://localhost:3000;
        proxy_set_header X-SSL-Client-S-DN $ssl_client_s_dn;
    }
}
```

#### 4. VPN-Only Access

```bash
# Deploy to private network only
# No public internet access
# Users must connect via VPN

# Firewall rules
sudo ufw default deny incoming
sudo ufw allow from 10.0.0.0/8  # VPN subnet only
sudo ufw enable
```

### Application Download Security

#### Secure Download Portal

```typescript
// pages/api/download/[app].ts
import { createClient } from '@supabase/supabase-js';

export default async function handler(req, res) {
  // 1. Check authentication
  const token = req.headers.authorization?.split(' ')[1];
  if (!token) return res.status(401).json({ error: 'Unauthorized' });

  // 2. Verify user has download permission
  const supabase = createClient(process.env.NEXT_PUBLIC_SUPABASE_URL, token);
  const { data: { user } } = await supabase.auth.getUser();

  if (!user || user.role !== 'technician') {
    return res.status(403).json({ error: 'Forbidden' });
  }

  // 3. Generate signed URL (expires in 1 hour)
  const { data, error } = await supabase.storage
    .from('apps')
    .createSignedUrl(`ppsp-${req.query.app}.dmg`, 3600);

  if (error) return res.status(500).json({ error });

  // 4. Log download
  await supabase.from('download_logs').insert({
    user_id: user.id,
    app: req.query.app,
    ip: req.socket.remoteAddress,
    downloaded_at: new Date().toISOString()
  });

  // 5. Redirect to signed URL
  res.redirect(302, data.signedUrl);
}
```

### USB Security Hardening

#### Tamper Detection
```bash
# Add to USB boot script
# Check for tampering

EXPECTED_HASH="sha256:abc123..."
CURRENT_HASH=$(sha256sum /boot/vmlinuz | cut -d' ' -f1)

if [ "$EXPECTED_HASH" != "$CURRENT_HASH" ]; then
    echo "TAMPERING DETECTED!"
    # Wipe sensitive data
    shred -vfz -n 3 /dev/sda3
    shutdown -h now
fi
```

#### Self-Destruct
```bash
# After 3 failed login attempts
FAILED_ATTEMPTS=$(cat /var/log/failed_logins | wc -l)

if [ $FAILED_ATTEMPTS -ge 3 ]; then
    # Wipe encryption keys
    cryptsetup luksErase /dev/sda2
    # Wipe data partition
    dd if=/dev/urandom of=/dev/sda3 bs=1M
    shutdown -h now
fi
```

### Audit Logging

```typescript
// lib/audit.ts
export async function logAudit(event: string, details: any) {
  await supabase.from('audit_logs').insert({
    timestamp: new Date().toISOString(),
    event,
    details,
    ip: await getClientIP(),
    user_agent: navigator.userAgent,
    session_id: getSessionId()
  });
}

// Usage
await logAudit('DRIVE_WIPE_STARTED', {
  device: '/dev/disk2',
  method: 'DoD-5220',
  technician: user.id
});
```

### Security Checklist

- [ ] Enable HTTPS only (HSTS)
- [ ] Add basic auth or IP whitelist
- [ ] Enable database RLS policies
- [ ] Set up audit logging
- [ ] Configure backup encryption
- [ ] Implement download tracking
- [ ] Add rate limiting
- [ ] Set up monitoring/alerting
- [ ] Regular security audits
- [ ] Employee access reviews

### Emergency Procedures

**If portal is compromised:**
1. Immediately rotate all API keys
2. Reset all user passwords
3. Review audit logs for unauthorized access
4. Notify affected customers
5. File incident report

**Contact:**
Security Team: security@ppsprecyclingllc.com  
Emergency: 916-381-8304 ext. 9
