# ALIEN ERP INTEGRATION GUIDE
## PPSP Recycling LLC - Data Destruction Module

### Overview
This guide explains how to integrate the PPSP Data Destruction certificates and reports with your Alien ERP system.

### Integration Methods

#### Method 1: REST API (Recommended)

**Endpoint Configuration:**
```
Base URL: https://alien.ppsprecyclingllc.com/api/v1/
Authentication: Bearer Token
Content-Type: application/json
```

**Certificate Upload:**
```javascript
POST /certificates
Headers:
  Authorization: Bearer {ALIEN_API_TOKEN}
  Content-Type: multipart/form-data

Body:
{
  "certificate_id": "PPSP-1709151234567",
  "pdf_file": [binary],
  "json_data": {
    "device_serial": "S3Z2NB0K1234567",
    "device_model": "Samsung SSD 860 EVO",
    "capacity_gb": 500,
    "wipe_method": "DoD 5220.22-M",
    "passes": 3,
    "duration_seconds": 2688,
    "technician_id": "TEC-001",
    "customer_id": "CUST-12345",
    "barcode_data": "PPSP-1709151234567",
    "verification_url": "https://ppsprecyclingllc.com/verify/PPSP-1709151234567"
  }
}
```

**Response:**
```json
{
  "success": true,
  "alien_record_id": "ALI-2024-001",
  "status": "recorded",
  "print_job_id": "PRJ-789"
}
```

#### Method 2: File Drop (Simple)

**Setup:**
1. Create shared network folder: `\\alien-server\ppsp-inbox\`
2. PPSP app saves files to this folder
3. Alien monitors folder and imports automatically

**File Naming Convention:**
```
{certificate_id}_{timestamp}.{ext}

Examples:
PPSP-1709151234567_20260228-143022.pdf
PPSP-1709151234567_20260228-143022.json
PPSP-1709151234567_20260228-143022.zpl (Zebra label)
```

**JSON Format:**
```json
{
  "ppsp_version": "2.0",
  "certificate_id": "PPSP-1709151234567",
  "barcode": "PPSP-1709151234567",
  "timestamp": "2026-02-28T14:30:22Z",
  "facility": {
    "name": "PPSP Recycling LLC",
    "location": "Elk Grove, CA",
    "contact": "916-381-8304"
  },
  "technician": {
    "id": "TEC-001",
    "name": "John Smith",
    "signature": "[digital signature]"
  },
  "customer": {
    "id": "CUST-12345",
    "name": "ABC Corporation",
    "po_number": "PO-2024-001"
  },
  "device": {
    "type": "SSD",
    "manufacturer": "Samsung",
    "model": "860 EVO",
    "serial_number": "S3Z2NB0K1234567",
    "capacity_bytes": 500107862016,
    "interface": "SATA III",
    "form_factor": "2.5 inch"
  },
  "destruction": {
    "method": "DoD 5220.22-M",
    "standard": "US DoD 5220.22-M",
    "passes": 3,
    "verification": "PASSED",
    "start_time": "2026-02-28T14:30:22Z",
    "end_time": "2026-02-28T15:20:33Z",
    "duration_seconds": 2688
  },
  "environmental": {
    "weight_lbs": 0.15,
    "materials": ["aluminum", "plastic", "silicon"],
    "recycling_code": "E-WASTE-SSD"
  },
  "compliance": {
    "nist_800_88": true,
    "dod_5220": true,
    "hipaa": true,
    "sox": true
  },
  "verification": {
    "url": "https://ppsprecyclingllc.com/verify/PPSP-1709151234567",
    "hash": "sha256:abc123..."
  }
}
```

#### Method 3: Email Integration

**Configuration:**
```javascript
const nodemailer = require('nodemailer');

const transporter = nodemailer.createTransporter({
  host: 'smtp.ppsprecyclingllc.com',
  auth: {
    user: 'certificates@ppsprecyclingllc.com',
    pass: process.env.EMAIL_PASS
  }
});

// Send to Alien
await transporter.sendMail({
  from: 'certificates@ppsprecyclingllc.com',
  to: 'alien-import@ppsprecyclingllc.com',
  subject: `Certificate ${certId}`,
  text: `Automated certificate upload`,
  attachments: [
    { filename: `${certId}.pdf`, path: pdfPath },
    { filename: `${certId}.json`, content: JSON.stringify(data) },
    { filename: `${certId}.zpl`, content: zebraLabel }
  ]
});
```

### Barcode Printing for Alien

#### Standard Label Format (4" x 6")
```
┌──────────────────────────────────────┐
│ [BARCODE: Code 128]                  │
│ PPSP-1709151234567                   │
│                                      │
│ PPSP Recycling LLC                   │
│ Certificate of Destruction           │
│                                      │
│ Device: Samsung SSD 860 EVO          │
│ S/N: S3Z2NB0K1234567                 │
│ Date: 02/28/2026                     │
│                                      │
│ [QR CODE]  Scan to verify            │
└──────────────────────────────────────┘
```

#### Zebra Printer Setup
```javascript
// ZPL Code for 4x6 label
const zplLabel = `
^XA
^PW800
^LL1200
^FO50,50
^BCN,150,Y,N,N
^FD${certId}^FS
^FO50,220
^A0N,40,40
^FDPPSP Recycling LLC^FS
^FO50,280
^A0N,30,30
^FDCertificate of Destruction^FS
^FO50,350
^A0N,25,25
^FDDevice: ${device.model}^FS
^FO50,400
^A0N,25,25
^FDS/N: ${device.serial}^FS
^FO50,450
^A0N,25,25
^FDDate: ${date}^FS
^FO500,600
^BQN,2,5
^FDQA,${verifyUrl}^FS
^XZ
`;
```

### Database Schema for Alien

```sql
-- Create table for PPSP certificates
CREATE TABLE ppsp_certificates (
    id SERIAL PRIMARY KEY,
    certificate_id VARCHAR(50) UNIQUE NOT NULL,
    barcode VARCHAR(50) NOT NULL,
    alien_record_id VARCHAR(50),

    -- Device info
    device_type VARCHAR(50),
    device_manufacturer VARCHAR(100),
    device_model VARCHAR(100),
    device_serial VARCHAR(100),
    capacity_gb DECIMAL(10,2),

    -- Destruction info
    wipe_method VARCHAR(50),
    passes INTEGER,
    duration_seconds INTEGER,
    verification_status VARCHAR(20),

    -- Technician
    technician_id VARCHAR(20),
    technician_name VARCHAR(100),

    -- Customer
    customer_id VARCHAR(20),
    customer_name VARCHAR(200),
    po_number VARCHAR(50),

    -- Timestamps
    destruction_date TIMESTAMP,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    -- Files
    pdf_path VARCHAR(500),
    json_path VARCHAR(500),

    -- Verification
    verification_url VARCHAR(500),
    verification_hash VARCHAR(100)
);

-- Index for fast lookups
CREATE INDEX idx_cert_id ON ppsp_certificates(certificate_id);
CREATE INDEX idx_device_serial ON ppsp_certificates(device_serial);
CREATE INDEX idx_customer ON ppsp_certificates(customer_id);
CREATE INDEX idx_date ON ppsp_certificates(destruction_date);
```

### Testing Integration

```bash
# Test API connection
curl -X POST https://alien.ppsprecyclingllc.com/api/v1/certificates   -H "Authorization: Bearer YOUR_TOKEN"   -H "Content-Type: application/json"   -d '{
    "certificate_id": "TEST-1234567890",
    "device_serial": "TEST123",
    "wipe_method": "DoD 5220.22-M"
  }'

# Expected response:
{"success": true, "alien_record_id": "ALI-TEST-001"}
```

### Troubleshooting

| Issue | Solution |
|-------|----------|
| Alien not receiving data | Check network connectivity, firewall rules |
| Barcode won't scan | Ensure Code 128 format, minimum 2" width |
| PDF not opening | Verify PDF version compatibility |
| Duplicate records | Check certificate_id uniqueness constraint |

### Support

**Integration Support:**  
Email: integration@ppsprecyclingllc.com  
Phone: 916-381-8304 ext. 2
